#include "GIRenderer.h"

GIRenderer::GIRenderer() : DefaultRenderer() {
    m_shaderName = "GIRenderer";
    m_textureNamePrefix = "GIRenderer::";

    pathTraceOptions.raysPerPixel = 1;
    pathTraceOptions.maxScatteringEvents = 2;

    m_matteTemporalFilterSettings.hysteresis    = 0.985f;
    m_glossyTemporalFilterSettings.hysteresis   = 0.985f;

    const ImageFormat* giImageFormat = ImageFormat::RGB16F();

    // 128x128 is a bogus initial size. It will be resized in renderIndirectIllumination
    m_rawMatteIndirect  = Texture::createEmpty("GIRenderer::indirect/matte/raw",  128, 128, giImageFormat);
    m_rawGlossyIndirect = Texture::createEmpty("GIRenderer::indirect/glossy/raw", 128, 128, ImageFormat::RGBA16F());

    m_rayFramebuffer =
        Framebuffer::create(
            Texture::createEmpty("GIRenderer::m_rayFramebuffer/0: ray origin", 128, 128, ImageFormat::RGB32F()),
            Texture::createEmpty("GIRenderer::m_rayFramebuffer/1: matte direction",  128, 128, ImageFormat::RGB16F()),
            Texture::createEmpty("GIRenderer::m_rayFramebuffer/2: glossy direction + pdf",  128, 128, ImageFormat::RGBA16F()));

    m_matteSpatialPrefilterOutputFramebuffer = Framebuffer::create(Texture::createEmpty("GIRenderer::m_matteSpatialPrefilterOutputFramebuffer", 128, 128, giImageFormat));
    m_matteSpatialPrefilterInputTexture = Texture::createEmpty("GIRenderer::indirect/matte/pre", 128, 128, giImageFormat);

    m_matteTemporalWeightBuffer     = Texture::createEmpty("GIRenderer::indirect/matte/temporal/weight", 128, 128, ImageFormat::R8());
    m_glossyTemporalWeightBuffer    = Texture::createEmpty("GIRenderer::indirect/glossy/temporal/weight", 128, 128, ImageFormat::R8());

    m_matteTemporalWeightHBlurBuffer = Texture::createEmpty("GIRenderer::indirect/matte/temporal/weightHBlur", 128, 128, ImageFormat::R8());
    m_glossyTemporalWeightHBlurBuffer = Texture::createEmpty("GIRenderer::indirect/glossy/temporal/weightHBlur", 128, 128, ImageFormat::R8());

    m_matteTemporalWeightFilteredBuffer = Texture::createEmpty("GIRenderer::indirect/matte/temporal/weightFiltered", 128, 128, ImageFormat::R8());
    m_glossyTemporalWeightFilteredBuffer = Texture::createEmpty("GIRenderer::indirect/glossy/temporal/weightFiltered", 128, 128, ImageFormat::R8());

    m_weightDilationFramebuffer = Framebuffer::create("GIRenderer::m_weightDilationFramebuffer");
    m_matteMedianFilterOutputFramebuffer = Framebuffer::create(Texture::createEmpty("GIRenderer::indirect/matte/median", 128, 128, giImageFormat));

    m_filteredMatteIndirectFramebuffer   = Framebuffer::create(Texture::createEmpty("GIRenderer::indirect/matte/filtered", 128, 128, giImageFormat));
    m_filteredGlossyIndirectFramebuffer  = Framebuffer::create(Texture::createEmpty("GIRenderer::indirect/glossy/filtered", 128, 128, giImageFormat));
    m_filteredMatteIndirectFramebuffer->texture(0)->visualization.max = 1.0f;

    m_pathTracer = PathTracer::create();
}


void GIRenderer::generateIndirectRays(RenderDevice* rd,const shared_ptr<GBuffer>& gbuffer, const shared_ptr<Framebuffer>& rayFramebuffer) {
    debugAssert(notNull(gbuffer->specification().encoding[GBuffer::Field::CS_NORMAL].format));
    rayFramebuffer->resize(rd->framebuffer());
    rd->push2D(rayFramebuffer); {
        rd->clear();
        rd->setDepthTest(RenderDevice::DEPTH_GREATER);
        Args args;
        gbuffer->setShaderArgsRead(args, "gbuffer_");
        args.setRect(rd->viewport());
        LAUNCH_SHADER("GIRenderer_generateRays.pix", args);
    } rd->pop2D();
}


void GIRenderer::copyIndirectRaysToCPU(const shared_ptr<Framebuffer>& rayFramebuffer, Array<Ray>& matteRay, Array<Ray>& glossyRay, Array<float>& glossyWeight) {
    BEGIN_PROFILER_EVENT("GIRenderer::copyIndirectRaysToCPU");

    BEGIN_PROFILER_EVENT("get GLPixelTransferBuffers");
    // Read the ray buffers back to the CPU
    const shared_ptr<GLPixelTransferBuffer>& originPTB = rayFramebuffer->texture(0)->toPixelTransferBuffer(ImageFormat::RGB32F());
    const shared_ptr<GLPixelTransferBuffer>& mattePTB  = rayFramebuffer->texture(1)->toPixelTransferBuffer(ImageFormat::RGB32F());
    const shared_ptr<GLPixelTransferBuffer>& glossyPTB = rayFramebuffer->texture(2)->toPixelTransferBuffer(ImageFormat::RGBA32F());
    END_PROFILER_EVENT();

    BEGIN_PROFILER_EVENT("Resize CPU buffers");
    // Generate one matte and one glossy ray per pixel
    const int N = originPTB->width() * originPTB->height();
    matteRay.resize(N);
    glossyRay.resize(N);  
    glossyWeight.resize(N);
    END_PROFILER_EVENT();

    BEGIN_PROFILER_EVENT("map GLPixelTransferBuffers");
    const Point3*  originPtr = (Point3*)originPTB->mapRead();
    const Vector3* mattePtr  = (Vector3*)mattePTB->mapRead();
    const Vector4* glossyPtr = (Vector4*)glossyPTB->mapRead();
    debugAssert(notNull(originPtr) && notNull(mattePtr) && notNull(glossyPtr));
    END_PROFILER_EVENT();

    BEGIN_PROFILER_EVENT("Multithreaded mapped memory copy");
    Thread::runConcurrently(0, N, [&](int i) {
        // Not a reference because originPtr is in GPU memory and
        // uncached...we *want* to copy it to the stack/registers
        const Point3  origin = originPtr[i];
        matteRay[i].set(origin, normalize(mattePtr[i]));

        const Vector4 glossyPacked = glossyPtr[i];
        glossyRay[i].set(origin, normalize(glossyPacked.xyz()));
        glossyWeight[i]  = glossyPacked.w;
    });
    END_PROFILER_EVENT();

    BEGIN_PROFILER_EVENT("unmap GLPixelTransferBuffers");
    glossyPTB->unmap();
    mattePTB->unmap();
    originPTB->unmap();
    END_PROFILER_EVENT();

    END_PROFILER_EVENT();
}


void GIRenderer::traceIndirectRays(int w, int h, const Array<Ray>& matteRay, const Array<Ray>& glossyRay, const Array<float>& glossyWeight, shared_ptr<GLPixelTransferBuffer>& matteLightPTB, shared_ptr<GLPixelTransferBuffer>& glossyLightPTB) {
    BEGIN_PROFILER_EVENT("GIRenderer::traceIndirectRays");
    debugAssert(w * h == matteRay.size());
    debugAssert(w * h == glossyRay.size());
    debugAssert(w * h == glossyWeight.size());

    // Ensure that the tree is built on the main thread
    // and GPU->CPU data transfer happens on the main thread
    m_pathTracer->prepare(pathTraceOptions);

    static Array<Radiance3> matteLight, glossyLight;
    static Array<float> glossyDistance;

    matteLight.resize(w * h);
    glossyLight.resize(w * h);
    glossyDistance.resize(w * h);

    const shared_ptr<Thread>& t = Thread::create("Trace", [&]() {
        BEGIN_PROFILER_EVENT("PathTrace matte");
        m_pathTracer->traceBuffer(m_matteRay, matteLight.getCArray(), pathTraceOptions, nullptr);
        END_PROFILER_EVENT();

        BEGIN_PROFILER_EVENT("PathTrace glossy");
        m_pathTracer->traceBuffer(m_glossyRay, glossyLight.getCArray(), pathTraceOptions, m_glossyWeight.getCArray(), glossyDistance.getCArray());
        END_PROFILER_EVENT();
    });

    t->start();

    // While tracing is running, communicate with OpenGL

    BEGIN_PROFILER_EVENT("create GLPixelTransferBuffers");
    matteLightPTB  = GLPixelTransferBuffer::create(w, h, ImageFormat::RGB32F());
    glossyLightPTB = GLPixelTransferBuffer::create(w, h, ImageFormat::RGBA32F());
    END_PROFILER_EVENT();


    BEGIN_PROFILER_EVENT("map GLPixelTransferBuffers");
    Radiance3* matteLightPtr  = (Radiance3*)matteLightPTB->mapWrite();
    Color4*    glossyLightPtr = (Color4*)glossyLightPTB->mapWrite();
    END_PROFILER_EVENT();

    t->waitForCompletion();
    if (updateAverageValues) {
        averageMatteValue = Color3(0.0f);
        for (auto c : matteLight) {
            averageMatteValue += c;
        }
        averageMatteValue /= float(matteLight.size());

        averageGlossyValue = Color3(0.0f);
        for (auto c : glossyLight) {
            averageGlossyValue += c;
        }
        averageGlossyValue /= float(glossyLight.size());
    }

    System::memcpy(matteLightPtr, matteLight.getCArray(), w * h * sizeof(Radiance3));
    Thread::runConcurrently(0, w * h, [&](int i) {
        glossyLightPtr[i] = Color4(glossyLight[i], glossyDistance[i]);
    });

    BEGIN_PROFILER_EVENT("unmap GLPixelTransferBuffers");
    glossyLightPTB->unmap();
    matteLightPTB->unmap();
    END_PROFILER_EVENT();

    END_PROFILER_EVENT();
}


void GIRenderer::renderIndirectIllumination(RenderDevice* rd, const shared_ptr<GBuffer>& gbuffer, const LightingEnvironment& environment) {
    if (! enableGI) { return; }

    const int w = gbuffer->width(), h = gbuffer->height();
    m_rawMatteIndirect->resize(w, h);
    m_rawGlossyIndirect->resize(w, h);

    if (updateTracedRays) {
        // There isn't much overhead (compared to the cost of ray tracing)
        // for periodically executing deleteAllBuffers() and that causes the profiler
        // to correctly note sync time.
        GLPixelTransferBuffer::deleteAllBuffers();
        generateIndirectRays(rd, gbuffer, m_rayFramebuffer);
        GLPixelTransferBuffer::deleteAllBuffers();
        copyIndirectRaysToCPU(m_rayFramebuffer, m_matteRay, m_glossyRay, m_glossyWeight);
        GLPixelTransferBuffer::deleteAllBuffers();

        shared_ptr<GLPixelTransferBuffer> matteLightPTB, glossyLightPTB;
        traceIndirectRays(w, h, m_matteRay, m_glossyRay, m_glossyWeight, matteLightPTB, glossyLightPTB);
        GLPixelTransferBuffer::deleteAllBuffers();

        copyIndirectLightToGPU(m_rawMatteIndirect, matteLightPTB, m_rawGlossyIndirect, glossyLightPTB);
        GLPixelTransferBuffer::deleteAllBuffers();
    }

    denoiseIndirectIllumination(rd, gbuffer);
}


void GIRenderer::setDeferredShadingArgs
   (Args&                               args, 
    const shared_ptr<GBuffer>&          gbuffer, 
    const LightingEnvironment&          environment) {
    DefaultRenderer::setDeferredShadingArgs(args, gbuffer, environment);

    args.setUniform("matteIndirectBuffer",  m_filteredMatteIndirectFramebuffer->texture(0), Sampler::buffer());
    args.setUniform("glossyIndirectBuffer", m_filteredGlossyIndirectFramebuffer->texture(0), Sampler::buffer());
}


void GIRenderer::onSceneLoad(const shared_ptr<Scene>& scene) {
    debugAssert(notNull(scene));
    m_pathTracer->setScene(scene);
}


void GIRenderer::copyIndirectLightToGPU
   (const shared_ptr<Texture>&                  matteLightTexture,
    const shared_ptr<GLPixelTransferBuffer>&    matteLightPTB, 
    const shared_ptr<Texture>&                  glossyLightTexture,
    const shared_ptr<GLPixelTransferBuffer>&    glossyLightPTB) {

    debugAssert(notNull(matteLightTexture));
    matteLightTexture->update(matteLightPTB);
    glossyLightTexture->update(glossyLightPTB);
}


void GIRenderer::denoiseIndirectIllumination(RenderDevice* rd, const shared_ptr<GBuffer>& gbuffer) {
    BEGIN_PROFILER_EVENT("GIRenderer::denoiseIndirectIllumination");
    if (! filteringEnabled) {
        // No filter bypass case
        m_filteredMatteIndirectFramebuffer->resize(gbuffer->width(), gbuffer->height());
        shared_ptr<Texture> temp = m_filteredMatteIndirectFramebuffer->texture(0);
        m_rawMatteIndirect->copyInto(temp);
        m_filteredMatteIndirectFramebuffer->set(Framebuffer::COLOR0, temp);

        m_filteredGlossyIndirectFramebuffer->resize(gbuffer->width(), gbuffer->height());
        temp = m_filteredGlossyIndirectFramebuffer->texture(0);
        m_rawGlossyIndirect->copyInto(temp);
        m_filteredGlossyIndirectFramebuffer->set(Framebuffer::COLOR0, temp);
        return;
    }

    shared_ptr<Texture> matteTemporalFilteredTexture;
    shared_ptr<Texture> glossyTemporalFilteredTexture;
    m_matteTemporalWeightBuffer->resize(gbuffer->width(), gbuffer->height());
    m_glossyTemporalWeightBuffer->resize(gbuffer->width(), gbuffer->height());
    {
        debugAssert(notNull(gbuffer->texture(GBuffer::Field::DEPTH_AND_STENCIL)));
        debugAssert(notNull(gbuffer->texture(GBuffer::Field::SS_POSITION_CHANGE)));
        
        matteTemporalFilteredTexture = m_matteTemporalFilter.apply(rd, gbuffer->camera(), 
            m_rawMatteIndirect,
            gbuffer->texture(GBuffer::Field::DEPTH_AND_STENCIL),
            gbuffer->texture(GBuffer::Field::SS_POSITION_CHANGE),
            Vector2(0, 0), m_matteTemporalWeightBuffer, 3,
            m_matteTemporalFilterSettings);
        matteTemporalFilteredTexture->setName("GIRenderer::indirect/matte/temporal");

        glossyTemporalFilteredTexture = m_glossyTemporalFilter.apply(rd, gbuffer->camera(), m_rawGlossyIndirect,
            gbuffer->texture(GBuffer::Field::DEPTH_AND_STENCIL),
            gbuffer->texture(GBuffer::Field::SS_POSITION_CHANGE),
            Vector2(0, 0), m_glossyTemporalWeightBuffer, 3,
            m_glossyTemporalFilterSettings);
        glossyTemporalFilteredTexture->setName("GIRenderer::indirect/glossy/temporal");
    }

    Sampler weightSampler(WrapMode::CLAMP, InterpolateMode::NEAREST_NO_MIPMAP);
    
    auto blurWeight = [&](shared_ptr<Texture> input, shared_ptr<Texture> output, int R, bool isVertical) {
        m_weightDilationFramebuffer->set(Framebuffer::COLOR0, output);
        rd->push2D(m_weightDilationFramebuffer); {
            Args args;
            args.setMacro("VERTICAL", isVertical);
            args.setMacro("R", R);
            input->setShaderArgs(args, "temporalWeight_", weightSampler);
            args.setRect(rd->viewport());
            LAUNCH_SHADER("weightDilation.*", args);
        } rd->pop2D();
    };
    
    int matteR = std::max(m_matteSpatialFilterSettings.radius*m_matteSpatialFilterSettings.lowWeightFilterExpansionFactor,
        m_matteSpatialPreFilterSettings.radius*m_matteSpatialPreFilterSettings.lowWeightFilterExpansionFactor);
    int glossyR = m_glossySpatialFilterSettings.radius*m_glossySpatialFilterSettings.lowWeightFilterExpansionFactor;
    
    if (m_matteSpatialPreFilterSettings.lowWeightFilterExpansionFactor > 1 ||
        m_matteSpatialFilterSettings.lowWeightFilterExpansionFactor > 1) {
        m_matteTemporalWeightHBlurBuffer->resize(gbuffer->width(), gbuffer->height());
        m_matteTemporalWeightFilteredBuffer->resize(gbuffer->width(), gbuffer->height());
        blurWeight(m_matteTemporalWeightBuffer, m_matteTemporalWeightHBlurBuffer, matteR, false);
        blurWeight(m_matteTemporalWeightHBlurBuffer, m_matteTemporalWeightFilteredBuffer, matteR, true);
    }

    if (m_glossySpatialFilterSettings.lowWeightFilterExpansionFactor > 1) {
        m_glossyTemporalWeightHBlurBuffer->resize(gbuffer->width(), gbuffer->height());
        m_glossyTemporalWeightFilteredBuffer->resize(gbuffer->width(), gbuffer->height());
        blurWeight(m_glossyTemporalWeightBuffer, m_glossyTemporalWeightHBlurBuffer, matteR, false);
        blurWeight(m_glossyTemporalWeightHBlurBuffer, m_glossyTemporalWeightFilteredBuffer, matteR, true);
    }


    
    static const bool useMattePreFilter = true;

    if (useMattePreFilter) {
        // Blur the temporal filter's internal texture

        m_matteSpatialPrefilterOutputFramebuffer->resize(gbuffer->width(), gbuffer->height());
        m_matteSpatialPrefilterOutputFramebuffer->set(Framebuffer::COLOR0, m_matteTemporalFilter.previousTexture());

        m_matteSpatialPreFilter.apply(rd, matteTemporalFilteredTexture,
            m_matteSpatialPrefilterOutputFramebuffer, gbuffer, m_matteSpatialPreFilterSettings, m_matteTemporalWeightFilteredBuffer);
    }

    // Spatial Filtering
    m_matteMedianFilterOutputFramebuffer->texture(0)->resize(matteTemporalFilteredTexture->width(), matteTemporalFilteredTexture->height());
    m_filteredMatteIndirectFramebuffer->texture(0)->resize(matteTemporalFilteredTexture->width(), matteTemporalFilteredTexture->height());
    m_filteredGlossyIndirectFramebuffer->texture(0)->resize(glossyTemporalFilteredTexture->width(), glossyTemporalFilteredTexture->height());

    shared_ptr<Texture> matteSpatialInput = matteTemporalFilteredTexture;
    if (useMedianPreSpatialMatte) {
        rd->push2D(m_matteMedianFilterOutputFramebuffer); {
            Args args;
            matteTemporalFilteredTexture->setShaderArgs(args, "T_", Sampler::buffer());
            args.setRect(rd->viewport());
            LAUNCH_SHADER("MedianFilter.*", args);
        } rd->pop2D();
        matteSpatialInput = m_matteMedianFilterOutputFramebuffer->texture(0);
    }

    m_matteSpatialFilter.apply(rd, matteSpatialInput,
        m_filteredMatteIndirectFramebuffer, gbuffer, m_matteSpatialFilterSettings, m_matteTemporalWeightFilteredBuffer);

    m_glossySpatialFilter.apply(rd, glossyTemporalFilteredTexture,
        m_filteredGlossyIndirectFramebuffer, gbuffer, m_glossySpatialFilterSettings, m_glossyTemporalWeightFilteredBuffer);
    
    END_PROFILER_EVENT();
}


void GIRenderer::updateFilterSettings
   (const BilateralFilterSettings2& matteSpatialPreSettings,
    const BilateralFilterSettings2& matteSpatialSettings,
    const BilateralFilterSettings2& glossySpatialSettings,
    const TemporalFilter2::Settings& matteTemporalSettings,
    const TemporalFilter2::Settings& glossyTemporalSettings) {
    m_matteSpatialPreFilterSettings = matteSpatialPreSettings;
    m_matteSpatialFilterSettings    = matteSpatialSettings;
    m_glossySpatialFilterSettings   = glossySpatialSettings;
    m_matteTemporalFilterSettings   = matteTemporalSettings;
    m_glossyTemporalFilterSettings  = glossyTemporalSettings;
}
