#pragma once
/**
\file BilateralFilter2.h

A somewhat generic bilateral filter that  offers several methods
of generating sample locations. (see BilateralFilterSettings2::passBreakdown)
G3D already has a BilateralFilter class, we extended it here to support the functionality
described in the paper.

\maintainer, Michael Mara

\created 2017-04-12
\edited  2017-04-12

*/
#include <G3D/G3DAll.h>

class BilateralFilterSettings2 {
public:
    /** 
        Filter radius in pixels. This will be multiplied by stepSize.
        Each 1D filter will have 2*radius + 1 taps. If set to 0, the filter is turned off.
    */
    int radius;

    /** How big one standard deviation of the Gaussian is as a fraction of the radius */
    float stddevFraction;

    /**
        Default is to step in 2-pixel intervals. This constant can be
        increased while radius decreases to improve
        performance at the expense of some dithering artifacts.

        Must be at least 1.
    */
    int stepSize;

    /** 
        If true, ensure that the "bilateral" weights are monotonically decreasing
        moving away from the current pixel. Default is true, only works doing a separable blur 
    */
    bool monotonicallyDecreasingBilateralWeights;

    /** Method for actually generating the taps; SINGLE_PASS is a brute force box filter (with appropriate weights,
        SPARSE generates taps in the same manner as G3D's ambient obsurance implementation */
    G3D_DECLARE_ENUM_CLASS(PassBreakdown, SINGLE_PASS, TWO_1D_PASSES, TWO_1D_PASSES_ALTERNATING, SPARSE, SPARSE_TWO_PASS);
    PassBreakdown passBreakdown;

    /** Number of taps to take in sparse pass */
    int sparseSampleCount;

    bool temporallyVarySparseSamples;

    /** weight for box filter added on top of gaussian */
    float weightOffset;

    /**
        How much depth difference is taken into account.
        Default is 1.
    */
    float depthWeight;

    /**
        How much normal difference is taken into account.
        Default is 1.
    */
    float normalWeight;

    /**
        How much plane difference is taken into account.
        Default is 1.
    */
    float planeWeight;

    /**
        How much glossy exponent is taken into account.
        Default is 1.
    */
    float glossyWeight;
    
    /** Use glossy to determine the radius */
    bool useGlossyForRadius;

    /** Use packed depth and normal together in a RGBA8 texture to save bandwidth */
    bool packedDepthAndNormal;

    /** Underweight large light values when averaging */
    bool fireflyReduction;
    
    /** How much bigger to make the filter if the weight was too low */
    int lowWeightFilterExpansionFactor;

    /** At what threshold do we increase the filter size */
    float filterExpansionThreshold;

    BilateralFilterSettings2();

    void makeGUI(GuiPane* pane);

};

/**
Multipass 2D bilateral filter
*/
class BilateralFilter2 {
protected:

    int m_runCount;

    shared_ptr<Framebuffer> m_intermediateFramebuffer;

    shared_ptr<Framebuffer> m_packedKeyBuffer;

    void setShaderArgs(UniformTable& args,RenderDevice* rd,
        const shared_ptr<Texture>&      source,
        const shared_ptr<Texture>&      weight,
        const shared_ptr<GBuffer>&      gbuffer,
        const BilateralFilterSettings2 & settings);

    void packBlurKeys(RenderDevice* rd,
        const shared_ptr<Texture>& cszBuffer,
        const float farPlaneZ,
        const shared_ptr<Texture>& normalBuffer);

public:
    /**
    Applies a Bilateral Filter. Handles intermediate storage in
    a texture of the same format as the source texture.
    */
    void apply(RenderDevice* rd, const shared_ptr<Texture>& source, const shared_ptr<Framebuffer>& destination, const shared_ptr<GBuffer>& gbuffer, const BilateralFilterSettings2& settings, const shared_ptr<Texture>& weight = shared_ptr<Texture>());

};
