#include "BilateralFilter2.h"

BilateralFilterSettings2::BilateralFilterSettings2() :
    radius(4),
    stddevFraction(0.4f),
    passBreakdown(PassBreakdown::SINGLE_PASS),
    sparseSampleCount(99),
    weightOffset(0.0f),
    temporallyVarySparseSamples(false),
    stepSize(1),
    monotonicallyDecreasingBilateralWeights(false),
    depthWeight(1.0f),
    normalWeight(1.0f),
    planeWeight(1.0f),
    glossyWeight(0.0f),
    useGlossyForRadius(false),
    packedDepthAndNormal(false),
    fireflyReduction(false),
    filterExpansionThreshold(0.5f), 
    lowWeightFilterExpansionFactor(1) {
}
void BilateralFilterSettings2::makeGUI(GuiPane * pane) {
    pane->addLabel("Bilateral Filter Settings 2");

    GuiTabPane* tabPane = pane->addTabPane(0);

    GuiPane* samplingPane = tabPane->addTab("Sampling");
    samplingPane->addEnumClassRadioButtons<BilateralFilterSettings2::PassBreakdown>("Pass Breakdown", &passBreakdown);
    samplingPane->addNumberBox("Blur Radius", &radius, "", GuiTheme::LINEAR_SLIDER, 0, 64);
    samplingPane->addNumberBox("Blur Step", &stepSize, "", GuiTheme::LINEAR_SLIDER, 1, 4);
    samplingPane->addNumberBox("Sparse Sample Count", &sparseSampleCount, "", GuiTheme::LINEAR_SLIDER, 3, 99);
    samplingPane->addCheckBox("Temporally Vary Samples", &temporallyVarySparseSamples);
    samplingPane->addCheckBox("Glossy Varying Radius", &useGlossyForRadius);
    samplingPane->addNumberBox("Filter Expansion Threshold", &filterExpansionThreshold, "", GuiTheme::LINEAR_SLIDER, 0.0f, 1.0f);
    samplingPane->addNumberBox("Expansion Factor", &lowWeightFilterExpansionFactor, "", GuiTheme::LINEAR_SLIDER, 1, 8);
    samplingPane->pack();

    GuiPane* reconstructionPane = tabPane->addTab("Reconstruction");
    reconstructionPane->addNumberBox("stddev Fraction", &stddevFraction, "", GuiTheme::LINEAR_SLIDER, 0.0f, 3.0f);
    reconstructionPane->addNumberBox("Weight Offset", &weightOffset, "", GuiTheme::LINEAR_SLIDER, 0.0f, 5.0f);
    reconstructionPane->addCheckBox("monotonicallyDecreasingBilateralWeights", &monotonicallyDecreasingBilateralWeights)->setEnabled(false);
    reconstructionPane->addNumberBox("Depth Weight",  &depthWeight,   "", GuiTheme::LINEAR_SLIDER, 0.0f, 5.0f);
    reconstructionPane->addNumberBox("Normal Weight", &normalWeight,  "", GuiTheme::LINEAR_SLIDER, 0.0f, 5.0f);
    reconstructionPane->addNumberBox("Plane Weight",  &planeWeight,   "", GuiTheme::LINEAR_SLIDER, 0.0f, 5.0f);
    reconstructionPane->addNumberBox("Glossy Weight", &glossyWeight,  "", GuiTheme::LINEAR_SLIDER, 0.0f, 5.0f);
    reconstructionPane->addCheckBox("Packed Depth And Normal", &packedDepthAndNormal);
    reconstructionPane->addCheckBox("Firefly Reduction", &fireflyReduction);
    reconstructionPane->pack();
    tabPane->pack();
}


// Taken from G3D's AmbientOcclusionSettings::numSpiralTurns()
static int numSpiralTurns(int numSamples)  {
#define NUM_PRECOMPUTED 100

    static int minDiscrepancyArray[NUM_PRECOMPUTED] = {
        //  0   1   2   3   4   5   6   7   8   9
        1,  1,  1,  2,  3,  2,  5,  2,  3,  2,  // 0
        3,  3,  5,  5,  3,  4,  7,  5,  5,  7,  // 1
        9,  8,  5,  5,  7,  7,  7,  8,  5,  8,  // 2
        11, 12,  7, 10, 13,  8, 11,  8,  7, 14,  // 3
        11, 11, 13, 12, 13, 19, 17, 13, 11, 18,  // 4
        19, 11, 11, 14, 17, 21, 15, 16, 17, 18,  // 5
        13, 17, 11, 17, 19, 18, 25, 18, 19, 19,  // 6
        29, 21, 19, 27, 31, 29, 21, 18, 17, 29,  // 7
        31, 31, 23, 18, 25, 26, 25, 23, 19, 34,  // 8
        19, 27, 21, 25, 39, 29, 17, 21, 27, 29 }; // 9

    if (numSamples < NUM_PRECOMPUTED) {
        return minDiscrepancyArray[numSamples];
    } else {
        return 5779; // Some large prime. Hope it does alright. It'll at least never degenerate into a perfect line until we have 5779 samples...
    }

#undef NUM_PRECOMPUTED


}


void BilateralFilter2::setShaderArgs(UniformTable & args, RenderDevice* rd,
    const shared_ptr<Texture>&      source,
    const shared_ptr<Texture>&      weight,
    const shared_ptr<GBuffer>&      gbuffer,
    const BilateralFilterSettings2 & settings) {
    args.setMacro("DEPTH_WEIGHT", settings.depthWeight);
    args.setMacro("NORMAL_WEIGHT", settings.normalWeight);
    args.setMacro("PLANE_WEIGHT", settings.planeWeight);
    args.setMacro("GLOSSY_WEIGHT", settings.glossyWeight);
    args.setMacro("GLOSSY_VARYING_RADIUS", settings.useGlossyForRadius);
    args.setMacro("R", settings.radius);
    args.setMacro("MDB_WEIGHTS", settings.monotonicallyDecreasingBilateralWeights ? 1 : 0);
    args.setUniform("stepSize", settings.stepSize);
    args.setUniform("stddevFrac", settings.stddevFraction);
    args.setUniform("source", source, Sampler::buffer());
    gbuffer->setShaderArgsRead(args, "gbuffer_");
    if (settings.packedDepthAndNormal) {
        m_packedKeyBuffer->texture(0)->setShaderArgs(args, "packedBilateralKey_", Sampler::buffer());
    }
    if (settings.filterExpansionThreshold > 0 && settings.lowWeightFilterExpansionFactor > 1 && weight) {
        args.setMacro("USE_EXPANSION_WEIGHT", true);
        args.setMacro("EXPANSION_FACTOR", settings.lowWeightFilterExpansionFactor);
        weight->setShaderArgs(args, "temporalWeight_", Sampler::buffer());
        args.setUniform("expansionThreshold", settings.filterExpansionThreshold);
    } else {
        args.setMacro("USE_EXPANSION_WEIGHT", false);
    }
    args.setUniform("weightOffset", settings.weightOffset);
    args.setMacro("ONE_D_PASS", settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::TWO_1D_PASSES
        || settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::TWO_1D_PASSES_ALTERNATING);
    
    bool sparsePass = settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::SPARSE ||
        settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::SPARSE_TWO_PASS;
    args.setMacro("SPARSE_PASS", sparsePass);
    if (sparsePass) {
        args.setMacro("TEMPORALLY_VARY_SAMPLES", settings.temporallyVarySparseSamples);
        args.setMacro("SPARSE_SAMPLES", settings.sparseSampleCount);
        args.setMacro("NUM_SPIRAL_TURNS", numSpiralTurns(settings.sparseSampleCount));
    }
    
    args.setMacro("FAR_PLANE_Z", -1000.0f);
    args.setMacro("PACKED_DEPTH_AND_NORMAL", settings.packedDepthAndNormal);
    args.setMacro("USE_HIGHLIGHT_COMPRESSION", settings.fireflyReduction);
}


void BilateralFilter2::packBlurKeys
    (RenderDevice* rd,
    const shared_ptr<Texture>& cszBuffer,
    const float farPlaneZ,
    const shared_ptr<Texture>& normalBuffer) {

    rd->push2D(m_packedKeyBuffer); {
        Args args;
        args.setMacro("FAR_PLANE_Z", farPlaneZ);
        cszBuffer->setShaderArgs(args, "csZ_", Sampler::buffer());
        normalBuffer->setShaderArgs(args, "normal_", Sampler::buffer());
        args.setRect(rd->viewport());
        LAUNCH_SHADER_WITH_HINT("AmbientOcclusion_packBilateralKey.pix", args, "BilateralFilter2::packBlurKeys");
    } rd->pop2D();
}


void BilateralFilter2::apply
(RenderDevice*                          rd,
    const shared_ptr<Texture>&          source,
    const shared_ptr<Framebuffer>&      destination,
    const shared_ptr<GBuffer>&          gbuffer,
    const BilateralFilterSettings2 &    settings,
    const shared_ptr<Texture>&          weight) {

    if (settings.radius > 0) {
        if (settings.packedDepthAndNormal) {
            if (isNull(m_packedKeyBuffer)) {
                m_packedKeyBuffer = Framebuffer::create(Texture::createEmpty("BilateralFilter2::m_packedKeyBuffer",
                    source->width(), source->height(), ImageFormat::RGBA16()));
            }
            m_packedKeyBuffer->resize(gbuffer->width(), gbuffer->height());
            
            packBlurKeys(rd, gbuffer->texture(GBuffer::Field::CS_Z), -1000.0f, gbuffer->texture(GBuffer::Field::CS_NORMAL));
        }
        if (isNull(m_intermediateFramebuffer) || m_intermediateFramebuffer->texture(0)->encoding() != source->encoding()) {
            m_intermediateFramebuffer = Framebuffer::create(Texture::createEmpty("BilateralFilter2::intermediate",
                source->width(), source->height(), source->encoding()));
            m_runCount = 0;
        }
        m_intermediateFramebuffer->resize(source->width(), source->height());

        if (settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::SINGLE_PASS
            || settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::SPARSE) {
            rd->push2D(destination); {
                Args args;
                setShaderArgs(args, rd, source, weight, gbuffer, settings);
                args.setRect(rd->viewport());
                LAUNCH_SHADER("BilateralFilter2.pix", args);
            } rd->pop2D();
        } else {
            bool vertFirst = (settings.passBreakdown == BilateralFilterSettings2::PassBreakdown::TWO_1D_PASSES) ||
                (m_runCount % 2 == 0);
            rd->push2D(m_intermediateFramebuffer); {
                Args args;
                setShaderArgs(args, rd, source, weight, gbuffer, settings);
                args.setMacro("VERTICAL", vertFirst);
                args.setRect(rd->viewport());
                LAUNCH_SHADER("BilateralFilter2.pix", args);
            } rd->pop2D();
            rd->push2D(destination); {
                Args args;
                setShaderArgs(args, rd, m_intermediateFramebuffer->texture(0), weight, gbuffer, settings);
                args.setMacro("VERTICAL", !vertFirst);
                args.setRect(rd->viewport());
                LAUNCH_SHADER("BilateralFilter2.pix", args);
            } rd->pop2D();

        }
        
        ++m_runCount;
    } else {
        Texture::copy(source, destination->texture(0));
    }
}

