/** \file App.cpp */
#include "App.h"
#include "GIRenderer.h"

G3D_START_AT_MAIN();

int main(int argc, const char* argv[]) {
    {
        G3DSpecification g3dSpec;
        g3dSpec.audio = false;
        initGLG3D(g3dSpec);
    }

    GApp::Settings settings(argc, argv);
    settings.window.caption             = argv[0];
    settings.window.width               = 1280; settings.window.height       = 720;
    settings.window.fullScreen          = false;
    settings.window.resizable           = ! settings.window.fullScreen;
    settings.window.framed              = ! settings.window.fullScreen;
    settings.window.asynchronous        = false;
    settings.hdrFramebuffer.depthGuardBandThickness = Vector2int16(0, 0);
    settings.hdrFramebuffer.colorGuardBandThickness = Vector2int16(0, 0);
    settings.hdrFramebuffer.sampleRateOneDimension = 
#   ifdef G3D_DEBUG
        0.1f;
#   else
        0.5f;
#   endif
    settings.dataDir                    = FileSystem::currentDirectory();
    settings.screenCapture.outputDirectory = "../../../journal/";

    settings.renderer.factory = &(GIRenderer::create);
    settings.renderer.deferredShading = true;
    settings.renderer.orderIndependentTransparency = true;

    return App(settings).run();
}


App::App(const GApp::Settings& settings) : GApp(settings), m_update3DImage(true) {
}


// Called before the application loop begins.  Load data here and
// not in the constructor so that common exceptions will be
// automatically caught.
void App::onInit() {
    GApp::onInit();
    setFrameDuration(1.0f / 60.0f);

    showRenderingStats      = false;

    m_gbufferSpecification.encoding[GBuffer::Field::CS_Z] = ImageFormat::RG32F();
    m_gbufferSpecification.encoding[GBuffer::Field::SS_POSITION_CHANGE] = ImageFormat::RG16F();

#   ifdef G3D_DEBUG
        m_exportSampleRateOneDimension = 0.5f;
#   endif

    makeGUI();

    developerWindow->videoRecordDialog->setCaptureGui(false);
    developerWindow->videoRecordDialog->setScreenShotFormat("PNG");
    developerWindow->videoRecordDialog->setHalfSize(false);
    developerWindow->videoRecordDialog->setQuality(2.5f);

    m_matteSpatialPreFilterSettings.radius = 6;
    m_matteSpatialPreFilterSettings.stepSize = 1;
    m_matteSpatialPreFilterSettings.monotonicallyDecreasingBilateralWeights = false;
    m_matteSpatialPreFilterSettings.lowWeightFilterExpansionFactor = 6;
    m_matteSpatialPreFilterSettings.fireflyReduction = true;
    m_matteSpatialPreFilterSettings.temporallyVarySparseSamples = true;
    m_matteSpatialPreFilterSettings.sparseSampleCount = 50;
    m_matteSpatialPreFilterSettings.passBreakdown = BilateralFilterSettings2::PassBreakdown::SPARSE_TWO_PASS;
    m_matteSpatialPreFilterSettings.packedDepthAndNormal = true;

    m_framesToConvergenceForExportResults = 250;

    m_matteSpatialFilterSettings.radius = 6;
    m_matteSpatialFilterSettings.stepSize = 1;
    m_matteSpatialFilterSettings.monotonicallyDecreasingBilateralWeights = false;
    m_matteSpatialFilterSettings.lowWeightFilterExpansionFactor = 6;
    m_matteSpatialFilterSettings.fireflyReduction = true;
    m_matteSpatialFilterSettings.sparseSampleCount = 50;
    m_matteSpatialFilterSettings.passBreakdown = BilateralFilterSettings2::PassBreakdown::SPARSE_TWO_PASS;
    m_matteSpatialFilterSettings.packedDepthAndNormal = true;
    
    
    m_glossySpatialFilterSettings.radius = 7;
    m_glossySpatialFilterSettings.monotonicallyDecreasingBilateralWeights = false;
    m_glossySpatialFilterSettings.stepSize = 1;
    m_glossySpatialFilterSettings.normalWeight = 5.0f;
    m_glossySpatialFilterSettings.glossyWeight = 5.0f;
    m_glossySpatialFilterSettings.stddevFraction = 0.5f;
    m_glossySpatialFilterSettings.useGlossyForRadius = true;
    m_glossySpatialFilterSettings.passBreakdown = BilateralFilterSettings2::PassBreakdown::TWO_1D_PASSES;

    m_matteTemporalFilterSettings.hysteresis = 0.985f;
    m_matteTemporalFilterSettings.rejectIfVeryDifferent = true;
    m_matteTemporalFilterSettings.saveWeightToBuffer = true;
    m_matteTemporalFilterSettings.useColorClipping = false;
    m_matteTemporalFilterSettings.useVirtualDistance = false;
    m_glossyTemporalFilterSettings.hysteresis = 0.985f;


    dynamic_pointer_cast<GIRenderer>(renderer())->useMedianPreSpatialMatte = true;



    // G3D never actually changes the Scene object, so we can just pass it once
    dynamic_pointer_cast<GIRenderer>(renderer())->onSceneLoad(scene());

    m_update3DImage = true;
    loadScene("G3D Sponza");
}


void App::exportAllResults() {
    // Generate all results for the paper using current settings
    Array<String> scenesToExport("G3D Fireplace Room", "G3D Sponza (Glossy)", "G3D Salle de Bain", "G3D Cornell Box", "G3D Living Room");
    for (int i = 0; i < scenesToExport.size(); ++i) {
        loadScene(scenesToExport[i]);
        setActiveCamera(scene()->typedEntity<Camera>("camera"));
        scene()->onSimulation(0.0);
        onExportResults();
    }
}


void App::onExportResults() {
    drawMessage("Exporting...");
    float oldSampleRate = m_exportSampleRateOneDimension;
    m_exportSampleRateOneDimension = 4;

    m_framebuffer->resize(Vector2int32(m_osWindowDeviceFramebuffer->vector2Bounds() * m_exportSampleRateOneDimension));
    GBuffer::Specification gbufferSpec = m_gbufferSpecification;
    extendGBufferSpecification(gbufferSpec);
    m_gbuffer->setSpecification(gbufferSpec);

    Array<shared_ptr<Surface>> surfaceArray;
    Array<shared_ptr<Surface2D>> ignore;
    onPose(surfaceArray, ignore);

    const shared_ptr<GIRenderer>& giRenderer = dynamic_pointer_cast<GIRenderer>(renderer());
    RenderDevice* rd = RenderDevice::current;


    const bool oldUpdateTracedRays = giRenderer->updateTracedRays;
    giRenderer->updateTracedRays = true;

    for (int i = 0; i < m_framesToConvergenceForExportResults; ++i) {
        m_gbuffer->resize(m_framebuffer->width(), m_framebuffer->height());
        m_gbuffer->prepare(renderDevice, activeCamera(), 0, -(float)previousSimTimeStep(), Vector2int16(0, 0), Vector2int16(0, 0));

        m_gbuffer->resize(m_framebuffer->width(), m_framebuffer->height());
        m_gbuffer->prepare(rd, activeCamera(), 0, -(float)previousSimTimeStep(), m_settings.hdrFramebuffer.depthGuardBandThickness, m_settings.hdrFramebuffer.colorGuardBandThickness);

        dynamic_pointer_cast<GIRenderer>(renderer())->updateFilterSettings(m_matteSpatialPreFilterSettings, m_matteSpatialFilterSettings, m_glossySpatialFilterSettings, m_matteTemporalFilterSettings, m_glossyTemporalFilterSettings);
        m_renderer->render(rd, m_framebuffer, scene()->lightingEnvironment().ambientOcclusionSettings.enabled ? m_depthPeelFramebuffer : nullptr,
            scene()->lightingEnvironment(), m_gbuffer, surfaceArray);
        
    }
    giRenderer->updateTracedRays = oldUpdateTracedRays;

    static shared_ptr<Texture> hdrResult = Texture::createEmpty("HDR Result Export Texture", m_framebuffer->texture(0)->width(), m_framebuffer->texture(0)->height(), ImageFormat::RGB32F());
    hdrResult->resize(m_framebuffer->texture(0)->width(), m_framebuffer->texture(0)->height());
    Texture::copy(m_framebuffer->texture(0), hdrResult);

    const shared_ptr<Image>& src = hdrResult->toImage();
    const shared_ptr<Image>& out = Image::create(int(src->width() / float(m_exportSampleRateOneDimension)), 
                                          int(src->height() / float(m_exportSampleRateOneDimension)),
                                          ImageFormat::RGB32F());
    Point2int32 outSize(out->width(), out->height());

    const int exportSampleRateOneDimension = int(m_exportSampleRateOneDimension);
    alwaysAssertM(m_exportSampleRateOneDimension > 1 && m_exportSampleRateOneDimension == exportSampleRateOneDimension,
        "Expected integer supersampling");
    const float sq = float(exportSampleRateOneDimension * exportSampleRateOneDimension);
    Thread::runConcurrently({ 0, 0 }, { out->width(), out->height() }, [&](Point2int32 P) {
        Color3 sum;
        for (Point2int32 offset(0,0); offset.y < exportSampleRateOneDimension; ++offset.y) {
            for (offset.x = 0; offset.x < exportSampleRateOneDimension; ++offset.x) {
                sum += src->get<Color3>(offset + P * exportSampleRateOneDimension);
            }
        }
        out->set(P, sum / sq);
    });

    out->save(FilePath::mangle(scene()->name()) + ".exr");

    m_update3DImage = true;
    m_exportSampleRateOneDimension = oldSampleRate;
}

// Make a functional, if not pretty, GUI
void App::makeGUI() {
    debugWindow->setVisible(true);
    developerWindow->videoRecordDialog->setEnabled(true);

    GuiPane* renderPane = debugPane->addPane("Rendering", GuiTheme::ORNATE_PANE_STYLE);
    renderPane->setNewChildSize(250);
    renderPane->addRadioButton("Manual (press R)", MANUAL, &m_renderRate);
    renderPane->addRadioButton("Intermittent", INTERMITTENT, &m_renderRate);
    renderPane->addRadioButton("Continous", CONTINUOUS, &m_renderRate);

    renderPane->addCheckBox("Update rays", &dynamic_pointer_cast<GIRenderer>(renderer())->updateTracedRays);

    renderPane->addNumberBox<float>("Supersample", &m_settings.hdrFramebuffer.sampleRateOneDimension, "x", GuiTheme::LOG_SLIDER, 0.1f, 16.0f)->setCaptionWidth(105);
    renderPane->addCheckBox("Denoise", &(dynamic_pointer_cast<GIRenderer>(renderer())->filteringEnabled));

    PathTracer::Options& pathTraceOptions = dynamic_pointer_cast<GIRenderer>(renderer())->pathTraceOptions;
    renderPane->addNumberBox<int>("Indirect bounces", &pathTraceOptions.maxScatteringEvents, "", GuiTheme::LINEAR_SLIDER, 1, 9)->setCaptionWidth(105);
    renderPane->addCheckBox("Enable evt. map", &pathTraceOptions.useEnvironmentMapForLastScatteringEvent);
    renderPane->addNumberBox("Max importance", &pathTraceOptions.maxImportanceSamplingWeight, "", GuiTheme::LOG_SLIDER, 0.5f, 100.0f);
    renderPane->addNumberBox("Max radiance", &pathTraceOptions.maxIncidentRadiance, "W/(m^2 sr)", GuiTheme::LOG_SLIDER, 1.0f, 1e6f);

    renderPane->pack();

    GuiPane* exportPane = debugPane->addPane("Export", GuiTheme::ORNATE_PANE_STYLE);
    exportPane->moveRightOf(renderPane);
    static float s = 8.0f;
    exportPane->addNumberBox<float>("Supersample", &m_exportSampleRateOneDimension, "x", GuiTheme::LOG_SLIDER, 0.1f, 16.0f);
    exportPane->addButton("Export", this, &App::onExportBuffers);
    exportPane->addNumberBox<int>("Warmup frames", &m_framesToConvergenceForExportResults, "", GuiTheme::LINEAR_SLIDER, 1, 300);
    exportPane->addButton("Export Results", this, &App::onExportResults);
    exportPane->addButton("Export All Results", this, &App::exportAllResults);
    exportPane->pack();

    {
        GuiTabPane* tabPane = debugPane->addTabPane();
        tabPane->moveRightOf(exportPane);

        GuiPane* mattePane = tabPane->addTab("Matte Filter");
        GuiPane* matteSpatialPrePane = mattePane->addPane("Spatial PreFilter");
        m_matteSpatialPreFilterSettings.makeGUI(matteSpatialPrePane);
        matteSpatialPrePane->pack();
        GuiPane* matteTemporalPane = mattePane->addPane("Temporal Filter");
        matteTemporalPane->moveRightOf(matteSpatialPrePane);
        m_matteTemporalFilterSettings.makeGui(matteTemporalPane);
        matteTemporalPane->pack();
        GuiPane* matteSpatialPane = mattePane->addPane("Spatial Filter");
        matteSpatialPane->moveRightOf(matteTemporalPane);
        m_matteSpatialFilterSettings.makeGUI(matteSpatialPane);
        matteSpatialPane->addCheckBox("Precede with Median", &(dynamic_pointer_cast<GIRenderer>(renderer())->useMedianPreSpatialMatte));
        matteSpatialPane->pack();
        mattePane->pack();

        GuiPane* glossyPane = tabPane->addTab("Glossy Filter");
        GuiPane* glossyTemporalPane = glossyPane->addPane("Temporal Filter");
        m_glossyTemporalFilterSettings.makeGui(glossyTemporalPane);
        glossyTemporalPane->pack();
        GuiPane* glossySpatialPane = glossyPane->addPane("Spatial Filter");
        glossySpatialPane->moveRightOf(glossyTemporalPane);
        m_glossySpatialFilterSettings.makeGUI(glossySpatialPane);
        glossySpatialPane->pack();
        glossyPane->pack();

        tabPane->pack();
    }

    debugWindow->pack();
    debugWindow->setRect(Rect2D::xywh(0, 0, (float)window()->width(), debugWindow->rect().height()));
}


void App::onGraphics3D(RenderDevice* rd, Array<shared_ptr<Surface> >& allSurfaces) {
    if (m_renderRate == CONTINUOUS) {
        m_update3DImage = true;
    } else if (m_renderRate == INTERMITTENT) {
        // Render once every three seconds
        static int count = 0;
        count = (count + 1) % (60 * 3);
        if (count == 0) {
            m_update3DImage = true;
        }
    }

    if (! scene()) {
        if ((submitToDisplayMode() == SubmitToDisplayMode::MAXIMIZE_THROUGHPUT) && (! rd->swapBuffersAutomatically())) {
            swapBuffers();
        }
        rd->clear();
        rd->pushState(); {
            rd->setProjectionAndCameraMatrix(activeCamera()->projection(), activeCamera()->frame());
            drawDebugShapes();
        } rd->popState();
        return;
    }

    GBuffer::Specification gbufferSpec = m_gbufferSpecification;
    extendGBufferSpecification(gbufferSpec);
    m_gbuffer->setSpecification(gbufferSpec);

    m_framebuffer->resize(m_settings.hdrFramebuffer.hdrFramebufferSizeFromOSWindowSize(Vector2int32(m_osWindowDeviceFramebuffer->vector2Bounds())));

    if (m_update3DImage) {
        m_gbuffer->resize(m_framebuffer->width(), m_framebuffer->height());
        m_gbuffer->prepare(rd, activeCamera(), 0, -(float)previousSimTimeStep(), m_settings.hdrFramebuffer.depthGuardBandThickness, m_settings.hdrFramebuffer.colorGuardBandThickness);

        dynamic_pointer_cast<GIRenderer>(renderer())->updateFilterSettings(m_matteSpatialPreFilterSettings, m_matteSpatialFilterSettings, m_glossySpatialFilterSettings, m_matteTemporalFilterSettings, m_glossyTemporalFilterSettings);
        m_renderer->render(rd, m_framebuffer, scene()->lightingEnvironment().ambientOcclusionSettings.enabled ? m_depthPeelFramebuffer : nullptr, 
            scene()->lightingEnvironment(), m_gbuffer, allSurfaces);

        // Debug visualizations and post-process effects
        rd->pushState(m_framebuffer); {
            // Call to make the App show the output of debugDraw(...)
            rd->setProjectionAndCameraMatrix(activeCamera()->projection(), activeCamera()->frame());
            drawDebugShapes();
            const shared_ptr<Entity>& selectedEntity = (notNull(developerWindow) && notNull(developerWindow->sceneEditorWindow)) ? developerWindow->sceneEditorWindow->selectedEntity() : nullptr;
            scene()->visualize(rd, selectedEntity, allSurfaces, sceneVisualizationSettings(), activeCamera());

            // Post-process special effects
            m_depthOfField->apply(rd, m_framebuffer->texture(0), m_framebuffer->texture(Framebuffer::DEPTH), activeCamera(), m_settings.hdrFramebuffer.depthGuardBandThickness - m_settings.hdrFramebuffer.colorGuardBandThickness);

            m_motionBlur->apply(rd, m_framebuffer->texture(0), m_gbuffer->texture(GBuffer::Field::SS_EXPRESSIVE_MOTION),
                                m_framebuffer->texture(Framebuffer::DEPTH), activeCamera(),
                                m_settings.hdrFramebuffer.depthGuardBandThickness - m_settings.hdrFramebuffer.colorGuardBandThickness);
        } rd->popState();

        m_update3DImage = false;
    }
    auto gir = dynamic_pointer_cast<GIRenderer>(renderer());

    // We're about to render to the actual back buffer, so swap the buffers now.
    // This call also allows the screenshot and video recording to capture the
    // previous frame just before it is displayed.
    if (submitToDisplayMode() == SubmitToDisplayMode::MAXIMIZE_THROUGHPUT) {
        swapBuffers();
    }

	// Clear the entire screen (needed even though we'll render over it, since
    // AFR uses clear() to detect that the buffer is not re-used.)
    rd->clear();

    // Perform gamma correction, bloom, and SSAA, and write to the native window frame buffer
    FilmSettings filmSettings = activeCamera()->filmSettings();
    // Disable vignetting because it intentionally adds noise
    filmSettings.setVignetteBottomStrength(0);
    filmSettings.setVignetteSizeFraction(0);
    filmSettings.setVignetteTopStrength(0);
    m_film->exposeAndRender(rd, filmSettings, m_framebuffer->texture(0), settings().hdrFramebuffer.colorGuardBandThickness.x + settings().hdrFramebuffer.depthGuardBandThickness.x, settings().hdrFramebuffer.depthGuardBandThickness.x);
}


bool App::onEvent(const GEvent& event) {
    if (GApp::onEvent(event)) {
        return true;
    }

    if ((event.type == GEventType::KEY_DOWN) && (event.key.keysym.sym == 'r')) {
        m_update3DImage = true;
    }

    return false;
}


void App::onAfterLoadScene(const Any &any, const String &sceneName) {
    // Force re-render
    m_update3DImage = true;

    // Override with appropriate AO settings for true indirect
    AmbientOcclusionSettings& ambientOcclusionSettings = scene()->lightingEnvironment().ambientOcclusionSettings;
    ambientOcclusionSettings.intensity = 0.5f;
    ambientOcclusionSettings.radius = 0.5f;
    ambientOcclusionSettings.blurRadius = 4;
    ambientOcclusionSettings.blurStepSize = 1;
    ambientOcclusionSettings.numSamples = 32;
    ambientOcclusionSettings.useDepthPeelBuffer = true;
    ambientOcclusionSettings.highQualityBlur = true;
    ambientOcclusionSettings.useNormalBuffer = true;
    ambientOcclusionSettings.useNormalsInBlur = true;
}

// For generating output buffers to use in other denoising techniques
void App::onExportBuffers() {
    drawMessage("Exporting...");

    String prefix = "export-";

    GBuffer::Specification gbufferSpec = m_gbufferSpecification;
    extendGBufferSpecification(gbufferSpec);
    gbufferSpec.encoding[GBuffer::Field::CS_POSITION] = ImageFormat::RGB32F();
    m_gbuffer->setSpecification(gbufferSpec);

    m_framebuffer->resize(Vector2int32(m_osWindowDeviceFramebuffer->vector2Bounds() * m_exportSampleRateOneDimension));
    m_gbuffer->resize(m_framebuffer->width(), m_framebuffer->height());
    m_gbuffer->prepare(renderDevice, activeCamera(), 0, -(float)previousSimTimeStep(), Vector2int16(0, 0), Vector2int16(0, 0));

    Array<shared_ptr<Surface>> surfaceArray;
    Array<shared_ptr<Surface2D>> ignore;
    onPose(surfaceArray, ignore);
        
    const shared_ptr<Framebuffer>& outputFramebuffer =
        Framebuffer::create(Texture::createEmpty("output radiance", m_gbuffer->width(), m_gbuffer->height(), ImageFormat::RGB32F()),
            m_gbuffer->texture(GBuffer::Field::DEPTH_AND_STENCIL));

    const shared_ptr<GIRenderer>& giRenderer = dynamic_pointer_cast<GIRenderer>(renderer());

    const bool oldFilteringEnabled = giRenderer->filteringEnabled;
    const bool oldUpdateTracedRays = giRenderer->updateTracedRays;

    giRenderer->updateTracedRays = true;
    renderDevice->pushState(outputFramebuffer); {
        renderDevice->clear();
    } renderDevice->popState();

    BilateralFilterSettings2 noSpatial;
    noSpatial.radius = 0;
    TemporalFilter2::Settings noTemporal;
    noTemporal.hysteresis = 0;

    TemporalFilter2::Settings noTemporal2;
    noTemporal2.hysteresis = 0;

    giRenderer->filteringEnabled = false;
    giRenderer->updateFilterSettings(noSpatial, noSpatial, noSpatial, noTemporal, noTemporal2);

    m_renderer->render(renderDevice, outputFramebuffer, scene()->lightingEnvironment().ambientOcclusionSettings.enabled ? m_depthPeelFramebuffer : nullptr, 
        scene()->lightingEnvironment(), m_gbuffer, surfaceArray);    

    m_gbuffer->texture(GBuffer::Field::CS_POSITION)->toImage(ImageFormat::RGB32F())->save(prefix + "position.exr");
    m_gbuffer->texture(GBuffer::Field::CS_NORMAL)->toImage(ImageFormat::RGB32F())->save(prefix + "normal.exr");
    m_gbuffer->texture(GBuffer::Field::LAMBERTIAN)->toImage(ImageFormat::RGB8())->save(prefix + "albedo.png");
    outputFramebuffer->texture(0)->toImage(ImageFormat::RGB32F())->save(prefix + "radiance.exr");
    giRenderer->filteringEnabled = oldFilteringEnabled;
    giRenderer->updateTracedRays = oldUpdateTracedRays;

    // Restore the screen
    m_update3DImage = true;
}