// -*- c++ -*-
/** \file GIRenderer_scatteredIndirectShade.pix */
#ifndef GIRenderer_scatteredIndirectShade_pix
#define GIRenderer_scatteredIndirectShade_pix

#include <g3dmath.glsl>
#include <Light/Light.glsl>
#include <LightingEnvironment/LightingEnvironment_LightUniforms.glsl>

/** Matte component of GI */
uniform sampler2D matteIndirectBuffer;

/** Glossy component of GI */
uniform sampler2D glossyIndirectBuffer;


Radiance3 computeScatteredIndirect(ivec2 C, UniversalMaterialSample surfel, Vector3 w_o, float AO) {
    // Incident W/m^2
    Irradiance3 matteLight  = texelFetch(matteIndirectBuffer, C, 0).rgb;
    Radiance3   glossyLight = texelFetch(glossyIndirectBuffer, C, 0).rgb;

    float glossyExponent = smoothnessToBlinnPhongExponent(surfel.smoothness);
    
    // For mirror reflection, w_h = n and cos_o = cos_i
    float cos_o = dot(surfel.glossyShadingNormal, w_o);

    // For mirror reflection (which is the only thing we can use for the evt light), the
    // half vector is the normal, so w_h . w_o == n . w_o == n . w_i. Push towards normal incidence
    // by using the sqrt of the dot product, to avoid over glossy reflection
    Color3 F = schlickFresnel(surfel.fresnelReflectionAtNormalIncidence, sqrt(max(0.001, cos_o)), surfel.smoothness);
    
    // Knock down F on broad lobes to account for the liklihood that the radiance
    // was not actually sampled at the mirror direction but a smaller part of that lobe
    // We experimented with but didn't ultimately use this term.
    // F *= square(min(1.0, 0.1 + surfel.smoothness));

    // m(w_i, w_o) = rho / pi
    Color3 lambertianColor = square(1.0 - F) * surfel.lambertianReflectivity * invPi;    

    // Slightly brighten to match path traced results; improves the approximation
    // of our fresnel term above.
    Color3 glossyColor = F * 1.2 / pi;
    
    // Because AO can only darken, restore some of the matte 
    // term lost to its baked in overblurred AO by upscaling lambertian indirect 5%.
    return (1.05 * AO) * pi * lambertianColor * matteLight + glossyColor * glossyLight;
}

#endif
